/**********************************************************
**  fsFileStream.hpp
**
**      fsFileStream is the object used to read and write
**	data from a file.
**
**	Gigawatt Studios
**	Copyright(C) 2000 - All Rights Reserved
\*********************************************************/

#ifdef FS_FILESTREAM_HPP
#error fsFileStream.hpp multiply included
#endif
#define FS_FILESTREAM_HPP

#include <string>

//--------------------------------------------------------------------------------
//	Forward references
//--------------------------------------------------------------------------------

struct fsFileStreamImp;

//--------------------------------------------------------------------------------
//--------------------------------------------------------------------------------
class fsFileStream
{
	public:

		//--------------------------------------------------------------------------------
		//	enum
		//--------------------------------------------------------------------------------
		enum AccessPointType
		{
			e_Beginning,
			e_Current,
			e_End,
		};

		enum AccessType
		{
			e_ReadOnly,
			e_WriteOnly,
			e_ReadWrite
		};

		//========================================================================
		//	structs
		//========================================================================

		//this header is preceded in the file by the four-byte signature "GWFB"
		struct Header
		{
			char m_Mode;
			char m_Major;
			char m_Minor;
			char m_Rev;
		};

		//========================================================================
		//	Constructor.  This could throw an exception if the file doesn't
		//	exist, or something.  So be ready! knowon exceptiosn thrown:
		//	fsInvalidLocatorX, fsFileDoesntExistX
		//========================================================================
		fsFileStream(const std::string& i_Locator, AccessType i_DesiredAccess);

		//========================================================================
		//	Destructor
		//========================================================================
		~fsFileStream();

		//========================================================================
		//	Read reads i_NumBytes into the buffer.  If the file is too short 
		//	to read	i_NumBytes, it will read as many as it can.  The return
		//	value is the number actually read into the o_Buffer.  This function
		//	also advances the file pointer.
		//========================================================================
		int Read(int i_NumBytes, void* o_Buffer);

		//========================================================================
		//	Write writes i_NumBytes from o_Buffer into the file.  The file 
		//	pointer will also be advanced.
		//========================================================================
		void Write(int i_NumBytes, const void* i_Buffer);

		//========================================================================
		//	GetFilePos returns the current position of the file pointer.
		//========================================================================
		int GetFilePos() const;

		//========================================================================
		//	SetFilePos sets the file pointer to the given value.  It is valid
		//	to move the file pointer beyond the end of the file.
		//========================================================================
		void SetFilePos( int i_Pos, AccessPointType i_DesiredAccessPoint = e_Beginning );

		//========================================================================
		//	GetLocator returns this filestream's associated locator
		//========================================================================
		const std::string& GetLocator() const;

		//========================================================================
		//	WriteHeader writes the given i_Header to the file. 
		//========================================================================
		void WriteHeader(const Header& i_Header);
		void WriteHeader(int i_Major = 0, int i_Minor = 0, int i_Rev = 0);

	private:

		fsFileStreamImp* m_pImp;
};

inline void fsFileStream::WriteHeader(int i_Major, int i_Minor, int i_Rev)
{
	Header h;
	h.m_Major = i_Major;
	h.m_Minor = i_Minor;
	h.m_Rev = i_Rev;
	WriteHeader(h);
}
